/** @file   hud.h
 * @brief   Declaraction of Hud - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_HUD_H
#define H_WWW_HUD_H

#include "eng2d.h"
#include <vector>
#include <string>
#include "weapon.h"
#include "objectivenote.h"

namespace WeWantWar {


/** @class  Hud
 * @brief   Manages the Heads-up-display
 * @author  Tomi Lamminsaari
 *
 */
class Hud
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** @struct OnScreenMessage
   * @brief   The messages showed to the player are instances of this struct
   * @author  Tomi Lamminsaari
   */
  struct OnScreenMessage {
    /** Counter telling how long this message is visible. If negative, the
     * message will not be shown and it will be removed from the table.
     */
    int counter;
    /** The message text. */
    std::string text;
  };
  
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Hud();
  
	/** Destructor
   */
	virtual ~Hud();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Hud
   */
  Hud( const Hud& rO );

	/** Assignment operator
   * @param     rO                Reference to another Hud
   * @return    Reference to us.
   */
  Hud& operator = ( const Hud& rO );

public:


  ///
  /// Methods
  /// =======

  /** Sets the cursor position
   * @param     rP                New position of the mouse cursor
   */
  void crosshairPos( const eng2d::Vec2D& rP );
  
  /** Sets the crosshair sprite
   * @param     aSprite           Pointer to crosshair sprite.
   */
  void setCrosshairSprite( RLE_SPRITE* aSprite );
  
  /** Sets the health level.
   * @param     hlevel            Health level ( 0 - 100 ).
   */
  void healthLevel( int hlevel );

  
  /** Sets the active weapon.
   * @param     wid               The weapon we should draw.
   */
  void activeWeapon( Weapon::WeaponID wid );
  
  /** Sets the ammocount.
   * @param     num               Number of ammunition there are left.
   */
  void ammocount( int num );
  
  /** Sets the number of grenades there are.
   * @param     num               Number of grenades
   */
  void grenadecount( int num );
  
  /** Sets the amount of bullettime left.
   * @param     bulltime          Amount of bullettime.
   */
  void bulletTime( int bulltime );

  /** Sets the amount of bodyarmor the player has.
   * @param   aBodyArmor        Amount of bodyarmor left
   * @param   aMaxBodyarmor     Maximum amount of bodyarmor
   */
  void setBodyArmor( int aBodyArmor, int aMaxBodyArmor );
  
  /** Adds new onscreen message
   * @param     txt               The text
   * @param     counter           How long the text should be shown
   */
  void addMessage( const std::string& txt, int counter );
  
  /** Sets the countdown timer
   * @param     v                 New timer value
   */
  void countdownValue( int v );
  
  /** Sets the objective - text
   * @param     rTxt              The text
   */
  void objectiveMessage( const std::string& rTxt );

  /** Sets the combo animation that we play.
   * @param   aComboAnim        Pointer to combo animation. Ownership is
   *                            transferred.
   */
  void setComboAnimation( eng2d::Animation* aComboAnim );


  /** Redraws the contents of the HUD
   * @param     pB                Pointer to target bitmap.
   */
  void redraw( BITMAP* pB );

  /** Updates the hud-elements
   */
  void update();
  
  
  
  ///
  /// Getter methods
  /// ==============
  
  /** Returns the pointer to weapon sprite that will be drawn on HUD.
   * @return  Pointer to rle sprite that represents the current weapon.
   */
  RLE_SPRITE* getWeaponSprite() const;
  


  ///
  /// Members
  /// =======

  /** The position of the mousecursor or weapon's sight. */
  eng2d::Vec2D m_cursorpos;
  
  /** Pointer to crosshair sprite. */
  RLE_SPRITE* m_crosshairSprite;
  
  /** The gradient we use when drawing the energybar. */
  eng2d::Gradient m_healthGrad;
  
  /** The countdown timer */
  int m_countdownTimer;
  
  /** Health level */
  int m_healthLevel;
  
  /** Number of ammunition */
  int m_ammocount;
  
  /** Number of grenades */
  int m_grenadecount;
  
  /** Amount of bullettime */
  int m_bullettime;
  
  /** The weapon ID */
  Weapon::WeaponID m_weaponID;
  
  /** The onscreen messages */
  std::vector< OnScreenMessage* >  m_messages;
  
  /** The Objective - message. */
  ObjectiveNote* m_pObjectiveMessage;

  /** The text what will be printed to the ObjectiveNote */
  std::string m_objectiveString;

  /** Pointer to combo animation */
  eng2d::Animation* m_comboAnimation;
  
  /** Amount of bodyarmor. */
  int m_bodyArmor;
  
  /** Maximum amount of bodyarmor */
  int m_maximumBodyArmor;
private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: hud.h,v $
 * Revision 1.4  2006/04/12 21:11:16  lamminsa
 * Bodyarmor added.
 *
 * Revision 1.3  2006/04/03 20:55:18  lamminsa
 * setComboAnimation() method added. m_comboAnimation member added.
 *
 * Revision 1.2  2006/02/08 22:03:28  lamminsa
 * New animation and graphic system was implemented.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:00+02  lamminsa
 * Initial revision
 *
 */
 
